/*
    This file is part of the LCDGraphics library for the Weebox.

    LCDGraphics is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Foobar is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with LCDGraphics.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "IDisplay.h"


#include "EP93xxDevice.h"
#include "HD61830Controller.h"
#ifdef WIN32
#include "WindowsController.h"
#endif
#include "IDevice.h"
#include "IController.h"
#include "Colour.h"
#include "ColourMaths.h"
#include "Rectangle.h"

#include <stdlib.h>
#include <algorithm>

namespace LCDGraphics
{
	class Display : public IDisplay
	{
	public:
		Display(IController* itController)
		{
			mController = itController;
			mColour = Colour(0, 0, 0);
			mScreen = new Colour[GetHeight()*GetWidth()];
			mUpdateCount = 0;
			mFlushNeeded = false;

			Clear(Colour(0xff, 0xff, 0xff));
		}
		~Display()
		{
			delete[] mScreen;
			delete mController;
		}

		int GetHeight() const
		{
			return mController->GetHeight();
		}

		int GetWidth() const
		{
			return mController->GetWidth();
		}

		int GetNumColourChannels() const
		{
			return mController->GetNumColourChannels();
		}

		int GetColourDepth() const
		{
			return mController->GetColourDepth();
		}

		void Clear(const Colour& iColour)
		{
			ClearClipRegion();
			int numPixels = GetHeight() * GetWidth();
			for (int i = 0; i < numPixels; ++i)
				mScreen[i] = iColour;
			Flush();
		}

		void SetColour(const Colour& iColour)
		{
			mColour = iColour;
		}

		Colour GetColour() const
		{
			return mColour;
		}

		void SetDrawMode(DrawMode iDrawMode)
		{
			mDrawMode = iDrawMode;
		}
		DrawMode GetDrawMode() const
		{
			return mDrawMode;
		}

		void DrawPixel(const Point& iPosition)
		{
			if (Plot(iPosition.GetX(), iPosition.GetY(), mColour))
				Flush();
		}

		void DrawLine(const Point& iFrom, const Point& iTo)
		{
			// Yay thanks Wikipedia, saved me working it out again from scratch: http://en.wikipedia.org/wiki/Bresenham's_line_algorithm
			int x0 = iFrom.GetX();
			int x1 = iTo.GetX();
			int y0 = iFrom.GetY();
			int y1 = iTo.GetY();

			const bool isSteep = abs(y1 - y0) > abs(x1 - x0);
			if (isSteep)
			{
				std::swap(x0, y0);
				std::swap(x1, y1);
			}
			if (x0 > x1)
			{
				std::swap(x0, x1);
				std::swap(y0, y1);
			}

			int deltaX = x1 - x0;
			int deltaY = abs(y1 - y0);
			int ystep = (y0 < y1) ? 1 : -1;
			int error = 0;
			int y = y0;
			for (int x = x0; x < x1; ++x)
			{
				if (isSteep)
					Plot(y, x, mColour);
				else
					Plot(x, y, mColour);
				error += deltaY;
				if ((error*2) >= deltaX)
				{
					y += ystep;
					error -= deltaX;
				}
			}
			Flush();
		}

		void DrawRegion(const Rectangle& iRectangle)
		{
			Rectangle clippedRectangle(iRectangle);
			clippedRectangle.ClipTo(mClipRegion);
			if (clippedRectangle.IsEmpty())
				return;

			for (int y = clippedRectangle.GetMinY(); y < clippedRectangle.GetMaxY(); ++y)
			{
				for (int x = clippedRectangle.GetMinX(); x < clippedRectangle.GetMaxX(); ++x)
				{
					mScreen[x + y * GetWidth()] = mColour;
				}
			}

			Flush();
		}

		void DrawRegionOutline(const Rectangle& iRectangle)
		{
			Rectangle clippedRectangle(iRectangle);
			clippedRectangle.ClipTo(mClipRegion);
			if (clippedRectangle.IsEmpty())
				return;

			for (int y = clippedRectangle.GetMinY(); y < clippedRectangle.GetMaxY(); ++y)
			{
				mScreen[clippedRectangle.GetMinX() + y * GetWidth()] = mColour;
				mScreen[clippedRectangle.GetMaxX() - 1 + y * GetWidth()] = mColour;
			}

			for (int x = clippedRectangle.GetMinX(); x < clippedRectangle.GetMaxX(); ++x)
			{
				mScreen[x + clippedRectangle.GetMinY() * GetWidth()] = mColour;
				mScreen[x + (clippedRectangle.GetMaxY()-1) * GetWidth()] = mColour;
			}
			Flush();
		}

		virtual void BeginUpdate()
		{
			++mUpdateCount;
		}

		virtual void EndUpdate()
		{
			if (mUpdateCount > 0)
			{
				--mUpdateCount;
				if (mUpdateCount == 0 && mFlushNeeded)
				{
					Flush();
				}
			}
		}

		virtual void DrawRaster1(const Point& iPosition, const unsigned char* iData, int iWidth, int iHeight, int iStride)
		{
			int topExtent = iPosition.GetY();
			int bottomExtent = iPosition.GetY() + iHeight;
			// Drop any bitmaps that are obviously off screen.
			if (bottomExtent < mClipRegion.GetMinY() || topExtent >= mClipRegion.GetMaxY())
				return;
			int leftExtent = iPosition.GetX();
			int rightExtent = iPosition.GetX() + iWidth;
			if (leftExtent >= mClipRegion.GetMaxX() || rightExtent <= mClipRegion.GetMinX())
				return;
			// Clip the extent to the screen.
			unsigned char startBitMask = 0x80;
			if (leftExtent < mClipRegion.GetMinX())
			{
				int pixelsToSkip = mClipRegion.GetMinX() - leftExtent;
				iData += (pixelsToSkip / 8);
				startBitMask >>= (pixelsToSkip % 8);
				leftExtent = mClipRegion.GetMinX();
			}
			if (rightExtent > mClipRegion.GetMaxX())
				rightExtent = mClipRegion.GetMaxX();
			if (topExtent < 0)
			{
				int linesToSkip = -topExtent;
				iData += linesToSkip * iStride;
				topExtent = 0;
			}
			if (bottomExtent > mClipRegion.GetMaxY())
				bottomExtent = mClipRegion.GetMaxY();
			if (bottomExtent > GetHeight())
				bottomExtent = GetHeight();

			int numPixelsPerLine = rightExtent - leftExtent;
			for (int y = topExtent; y < bottomExtent; ++y)
			{
				Colour* outPtr = &mScreen[leftExtent + GetWidth() * y];
				const unsigned char* inPtr = iData;
				unsigned char bitMask = startBitMask;
				unsigned char curByte = *inPtr++;
				for (int pixelCount = 0; pixelCount < numPixelsPerLine; ++pixelCount)
				{
					if (curByte & bitMask)
					{
						*outPtr++ = mColour;
					}
					else
						++outPtr;
					bitMask >>= 1;
					if (bitMask == 0)
					{
						bitMask = 0x80;
						curByte = *inPtr++;
					}
				}
				iData += iStride;
			}

			Flush();
		}


		virtual void DrawRaster8(const Point& iPosition, const unsigned char* iData, int iWidth)
		{
			// Drop any lines obviously off screen.
			if (iPosition.GetY() < mClipRegion.GetMinY() || iPosition.GetY() >= mClipRegion.GetMaxY())
				return;
			int startExtent = iPosition.GetX();
			int endExtent = iPosition.GetX() + iWidth;
			if (startExtent >= mClipRegion.GetMaxX() || endExtent <= mClipRegion.GetMinX())
				return;
			// Clip the extent to the screen.
			if (startExtent < mClipRegion.GetMinX())
			{
				iData += mClipRegion.GetMinX() - startExtent;
				startExtent = mClipRegion.GetMinX();
			}
			if (endExtent > mClipRegion.GetMaxX())
				endExtent = mClipRegion.GetMaxX();

			Colour* outPtr = &mScreen[startExtent + GetWidth() * iPosition.GetY()];

			while (startExtent < endExtent)
			{
				int colourValue = *iData++;
				if (colourValue > 0)
				{
					*outPtr++ = Colour(colourValue, colourValue, colourValue) * mColour;
				}
				else
					++outPtr;
				++startExtent;
			}

			Flush();
		}

		void ClearClipRegion()
		{
			mClipRegion = Rectangle(Point(0,0), GetWidth(), GetHeight());
		}

		void SetClipRegion(const Rectangle& iRectangle)
		{
			mClipRegion = iRectangle;
			mClipRegion.ClipTo(Rectangle(Point(0,0), GetWidth(), GetHeight()));
		}

		void GetClipRegion(Rectangle& oRectangle) const
		{
			oRectangle = mClipRegion;
		}


	private:
		void Flush()
		{
			if (mUpdateCount == 0)
			{
				mController->SetScreen(mScreen);
				mFlushNeeded = false;
			}
			else
				mFlushNeeded = true;
		}

		bool Plot(int iX, int iY, Colour c)
		{
			if (!mClipRegion.Contains(Point(iX, iY)))
				return false;

			mScreen[iX + iY * GetWidth()] = c;
			return true;
		}

	private:
		IController* mController;
		Colour mColour;
		DrawMode mDrawMode;
		Colour* mScreen;
		int mUpdateCount;
		bool mFlushNeeded;
		Rectangle mClipRegion;
	};

	// TODO: put elsewhere really.
	extern IDisplay* CreateDisplay()
	{
#ifdef WIN32
		IController* controller = CreateWindowsController();
#else
		IDevice* device = CreateEP93XXDevice();
		IController* controller = CreateHD61830Controller(device);
#endif
		return new Display(controller);
	}
}
