/*
    This file is part of the LCDGraphics library for the Weebox.

    LCDGraphics is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Foobar is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with LCDGraphics.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "EP93xxDevice.h"


#include "IDevice.h"
#include <cassert>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <iostream>


namespace LCDGraphics
{
	class EP93xxDevice : public IDevice
	{
	public:
		EP93xxDevice()
		{
			mDevMemFile = 0;
			mGPIO = 0;
			mBusy = false;
		}
		~EP93xxDevice()
		{
			Close();
		}

		bool Open()
		{
			if (mDevMemFile)
				return false;
			mDevMemFile = open("/dev/mem", O_RDWR | O_SYNC);
			if (!mDevMemFile)
				return false;
			mGPIO = mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, mDevMemFile, 0x80840000);
			mBusy = true;
			
			// Callibrate the delay loop.
			Callibrate();

			// Initialise port H bits as outputs.
			unsigned char portHDir = ReadIO(PORT_H_DIR);
			portHDir |= LCD_ENABLE | LCD_INSTRUCTION | LCD_READWRITE;
			WriteIO(PORT_H_DIR, portHDir);

			// Initialise port B bit 1 as an output and turn it on.
			WriteIO(PORT_B_DIR, ReadIO(PORT_B_DIR) | LCD_NEGVOLTAGE_ENABLE);
			WriteIO(PORT_B_DATA, ReadIO(PORT_B_DATA) | LCD_NEGVOLTAGE_ENABLE);
		}

		void Close()
		{
			// Shutdown the LCD power.
			WriteIO(PORT_B_DATA, ReadIO(PORT_B_DATA) &~ LCD_NEGVOLTAGE_ENABLE);

			if (mGPIO)
				munmap(mGPIO, getpagesize());
			mGPIO = 0;
			if (mDevMemFile)
				close(mDevMemFile);
			mDevMemFile = 0;
		}

		volatile unsigned int* GetIOAddress(int iPort) const
		{
			assert(mGPIO != 0);
			if (mGPIO == 0)
				return 0;
			return reinterpret_cast<volatile unsigned int*>(reinterpret_cast<char*>(mGPIO) + iPort);
		}

		void WriteIO(int iPort, unsigned char iValue) const
		{
			volatile unsigned int* port = GetIOAddress(iPort);
			if (port)
				*port = iValue;
		}

		unsigned char ReadIO(int iPort) const
		{
			volatile unsigned int* port = GetIOAddress(iPort);
			if (port)
				return static_cast<unsigned char>(*port);

			return 0;
		}

		void Callibrate()
		{
			NANOSECONDS_PER_LOOP = 1;
			const unsigned long numTestIterations = 10000000;
			struct timeval startTime;
			struct timeval endTime;
			gettimeofday(&startTime, NULL);
			Wait(numTestIterations);
			gettimeofday(&endTime, NULL);
			unsigned long timePassedInMicroseconds = (endTime.tv_sec - startTime.tv_sec) * 1000 * 1000
				+ (endTime.tv_usec - startTime.tv_usec);
			NANOSECONDS_PER_LOOP = static_cast<int>(timePassedInMicroseconds / (numTestIterations/1000));
		}

		void Wait(int iNanoseconds)
		{
			int numLoops = iNanoseconds / NANOSECONDS_PER_LOOP;
			asm volatile ("1:\n"
				"subs %1, %1, #1;\n"
				"bne 1b;\n"
			: "=r" (numLoops) : "r" (numLoops)
			);
		}

		bool WaitTilNotBusy()
		{
			if (!mBusy)
				return true;

			// Set port A to all inputs
			WriteIO(PORT_A_DIR, 0);

			// Read the current control registers.
			unsigned char control = ReadIO(PORT_H_DATA);
			unsigned char data;

			for (int counter = 0; counter < 1000; ++counter)
			{
				// RS = 1 (instruction), R/W = 1 (read)
				control |= LCD_INSTRUCTION | LCD_READWRITE;
				WriteIO(PORT_H_DATA, control);

				// Wait for the setup time.
				Wait(140);

				// Set the enable flag.
				control |= LCD_ENABLE;
				WriteIO(PORT_H_DATA, control);

				// Wait for data to be ready on the input bus.
				Wait(225);

				// Read the data.
				data = ReadIO(PORT_A_DATA);

				// Disable the chip.
				control &= ~LCD_ENABLE;
				WriteIO(PORT_H_DATA, control);

				// Wait for the enable to have been low long enough.
				Wait(450);

				if (!(data & 0x80))
				{
					mBusy = false;
					return true;
				}
			}

			std::cerr << "Gave up reading port: " << (int)data << std::endl;

			return false;
		}

		// ------ From IDevice.
		virtual bool SetInstruction(unsigned char iInstruction)
		{
			if (!mGPIO)
				return false;
			bool ok = WaitTilNotBusy();
			if (!ok)
				return ok;
			assert(!mBusy);

			// Set port A to all outputs.
			WriteIO(PORT_A_DIR, 0xff);

			// Adjust the control to assert the instruction register and not the read/write register.
			unsigned char control = ReadIO(PORT_H_DATA);
			control &= ~LCD_READWRITE;
			control |= LCD_INSTRUCTION;
			WriteIO(PORT_H_DATA, control);

			// Write the instruction
			WriteIO(PORT_A_DATA, iInstruction);

			// Wait for the setup time.
			Wait(140);

			// Assert the enable flag.
			control |= LCD_ENABLE;
			WriteIO(PORT_H_DATA, control);
			Wait(450);

			// De-assert the enable flag and wait for the low pulse amount.
			control &= ~LCD_ENABLE;
			WriteIO(PORT_H_DATA, control);
			Wait(450);

			// No need to set busy flag after setting the instruction register.
			return true;
		}

		virtual bool SetData(unsigned char iData)
		{
			if (!mGPIO)
				return false;
			return SetData(&iData, 1);
		}

		virtual bool SetData(const unsigned char* iData, unsigned int iDataCount)
		{
			if (!mGPIO)
				return false;

			for (; iDataCount > 0; ++iData, --iDataCount)
			{
				bool ok = WaitTilNotBusy();
				if (!ok)
					return ok;
				assert(!mBusy);

				// Set port A to all outputs.
				WriteIO(PORT_A_DIR, 0xff);

				// Adjust the control to de-assert the instruction register and read/write register.
				unsigned char control = ReadIO(PORT_H_DATA);
				control &= ~(LCD_READWRITE|LCD_INSTRUCTION);
				WriteIO(PORT_H_DATA, control);

				// Write the instruction
				WriteIO(PORT_A_DATA, *iData);

				// Wait for the setup time.
				Wait(140);

				// Assert the enable flag.
				control |= LCD_ENABLE;
				WriteIO(PORT_H_DATA, control);
				Wait(450);

				// De-assert the enable flag and wait for the low pulse amount.
				control &= ~LCD_ENABLE;
				WriteIO(PORT_H_DATA, control);
				Wait(450);

				mBusy = true;
			}
			return true;
		}

		virtual bool GetData(unsigned char* oData, unsigned int iDataCount)
		{
			if (!mGPIO)
				return false;

			for (; iDataCount > 0; ++oData, --iDataCount)
			{
				bool ok = WaitTilNotBusy();
				if (!ok)
					return ok;
				assert(!mBusy);

				// Set port A to all inputs.
				WriteIO(PORT_A_DIR, 0x00);

				// Adjust the control to de-assert the instruction register and assert the read/write register.
				unsigned char control = ReadIO(PORT_H_DATA);
				control &= ~LCD_INSTRUCTION;
				control |= LCD_READWRITE;
				WriteIO(PORT_H_DATA, control);

				// Wait for the setup time.
				Wait(140);

				// Assert the enable flag.
				control |= LCD_ENABLE;
				WriteIO(PORT_H_DATA, control);
				Wait(450);

				// Read the data
				*oData = ReadIO(PORT_A_DATA);

				// De-assert the enable flag and wait for the low pulse amount.
				control &= ~LCD_ENABLE;
				WriteIO(PORT_H_DATA, control);
				Wait(450);

				mBusy = true;
			}
			return true;
		}

	private:
		int mDevMemFile;
		void* mGPIO;
		bool mBusy;

		static const unsigned int PORT_A_DATA = 0x00;
		static const unsigned int PORT_A_DIR = 0x10;
		static const unsigned int PORT_B_DATA = 0x04;
		static const unsigned int PORT_B_DIR = 0x14;
		static const unsigned int PORT_F_DATA = 0x30;
		static const unsigned int PORT_F_DIR = 0x34;
		static const unsigned int PORT_H_DATA = 0x40;
		static const unsigned int PORT_H_DIR = 0x44;

		static const unsigned char LCD_ENABLE = (1<<3);
		static const unsigned char LCD_INSTRUCTION = (1<<4);
		static const unsigned char LCD_READWRITE = (1<<5);

		static const unsigned char LCD_NEGVOLTAGE_ENABLE = (1<<1);

		static int NANOSECONDS_PER_LOOP;
	};

	int EP93xxDevice::NANOSECONDS_PER_LOOP = 1;

	IDevice* CreateEP93XXDevice()
	{
		EP93xxDevice* device = new EP93xxDevice;
		if (device->Open())
			return device;

		delete device;
		return 0;
	}
}
