/*
    This file is part of the LCDGraphics library for the Weebox.

    LCDGraphics is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Foobar is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with LCDGraphics.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "IFontManager.h"

#include "Point.h"
#include "IDisplay.h"
#include "Rectangle.h"

#include <iostream>
#include <ft2build.h>
#include FT_FREETYPE_H
#include FT_CACHE_MANAGER_H
#include FT_CACHE_CHARMAP_H
#include FT_CACHE_SMALL_BITMAPS_H

#include <list>
#include <string>

namespace LCDGraphics
{
	class FontManager;

	class Font : public IFont
	{
	public:
		Font(FontManager* iManager, FTC_FaceID iID, int iYOffset, int iLineHeight)
		{
			mManager = iManager;
			mFaceID = iID;
			mYOffset = iYOffset;
			mLineHeight = iLineHeight;
		}
		~Font()
		{
			// No need to free anything; the magic inside the cache system does this.
		}

		virtual void Render(IDisplay* iDisplay, const char* iText, const Point& iStartPos);

		virtual void GetBoundingBox(const char* iText, const Point& iStartPos, Rectangle& oBounds) const;

		virtual int Font::GetLineGapInPixels() const;

		void DrawBitmap(LCDGraphics::IDisplay* iDisplay, FTC_SBit iBitmap, int iXPos, int iYPos)
		{
			const unsigned char* data = reinterpret_cast<const unsigned char*>(iBitmap->buffer);
			iDisplay->DrawRaster1(Point(iXPos, iYPos), data, iBitmap->width, iBitmap->height, iBitmap->pitch);
		}

	private:
		FTC_FaceID mFaceID;
		FontManager* mManager;
		int mYOffset;
		int mLineHeight;
	};

	class FontManager : public IFontManager
	{
	public:
		FontManager()
		{
			mLibrary = 0;
			mCacheManager = 0;
		}

		~FontManager()
		{
			if (mCacheManager)
				FTC_Manager_Done(mCacheManager);
			if (mLibrary)
				FT_Done_FreeType(mLibrary);
		}

		bool Initialise()
		{
			int error = FT_Init_FreeType(&mLibrary); 
			if (error)
				return false;

			error = FTC_Manager_New(mLibrary, 0, 0, 0, RequesterThunk, this, &mCacheManager);
			if (error)
				return false;

			error = FTC_CMapCache_New(mCacheManager, &mCharMapCache);
			if (error)
				return false;

			error = FTC_SBitCache_New(mCacheManager, &mBitmapCache);
			if (error)
				return false;

			return true;
		}

		virtual IFont* CreateFont(const char* iFilename, int iFontSize)
		{
			FTC_FaceID newID;
			int yOffset, height;
			if (!FindFace(iFilename, iFontSize, newID, yOffset, height))
				return 0;

			return new Font(this, newID, yOffset, height);
		}

		FT_Library GetLibrary() const
		{ 
			return mLibrary;
		}

		FTC_SBitCache GetBitmapCache() const
		{
			return mBitmapCache;
		}

		FTC_Manager GetCacheManager() const
		{
			return mCacheManager;
		}

		FTC_CMapCache GetCharMapCache() const
		{
			return mCharMapCache;
		}

		static FT_Error RequesterThunk(FTC_FaceID iFaceId, FT_Library iLibrary, FT_Pointer iRequestData, FT_Face* oFace)
		{
			return reinterpret_cast<FontManager*>(iRequestData)->Requester(iFaceId, oFace);
		}

		FT_Error Requester(FTC_FaceID iFaceId, FT_Face* oFace)
		{
			FontRecord* record = reinterpret_cast<FontRecord*>(iFaceId);

			int error = FT_New_Face(mLibrary, record->mName.c_str(), 0, oFace);
			if (error)
				return error;

			int newHeight = DeriveDesignHeightFromMaxHeight(*oFace, record->mSize);

			error = FT_Set_Pixel_Sizes(*oFace, 0, newHeight);
			if (error)
				return error;

			record->mYOffset = (newHeight * (*oFace)->ascender) / (*oFace)->units_per_EM;
			record->mHeight = (newHeight * ((*oFace)->ascender - (*oFace)->descender)) / (*oFace)->units_per_EM;
			return 0;
		}

		// Taken from http://www.mail-archive.com/freetype@nongnu.org/msg00123.html to allow us
		// to specify a true pixel height and not a "design height".
		static int DeriveDesignHeightFromMaxHeight(const FT_Face& aFace, int aMaxHeightInPixel)
		{
			const int boundingBoxHeightInFontUnit = aFace->bbox.yMax - aFace->bbox.yMin;
			int designHeightInPixels = ( ( aMaxHeightInPixel *
				aFace->units_per_EM ) / boundingBoxHeightInFontUnit );

			const int maxHeightInFontUnit = aMaxHeightInPixel << 6;
			FT_Set_Pixel_Sizes( aFace, designHeightInPixels, designHeightInPixels );
			int currentMaxHeightInFontUnit = FT_MulFix(
				boundingBoxHeightInFontUnit, aFace->size->metrics.y_scale );
			while ( currentMaxHeightInFontUnit < maxHeightInFontUnit )
			{
				designHeightInPixels++;
				FT_Set_Pixel_Sizes( aFace, designHeightInPixels, designHeightInPixels );
				currentMaxHeightInFontUnit = FT_MulFix(
					boundingBoxHeightInFontUnit, aFace->size->metrics.y_scale );
			}
			while ( currentMaxHeightInFontUnit > maxHeightInFontUnit )
			{
				designHeightInPixels--;
				FT_Set_Pixel_Sizes( aFace, designHeightInPixels, designHeightInPixels );
				currentMaxHeightInFontUnit = FT_MulFix(
					boundingBoxHeightInFontUnit, aFace->size->metrics.y_scale );
			}
			return designHeightInPixels;
		}

		bool FindFace(const char* iFilename, int iSize, FTC_FaceID& oFaceId, int& oOffset, int& oHeight)
		{
			FontRecord* record = FindFont(iFilename, iSize);
			if (record == 0)
			{
				FontRecord newRecord;
				newRecord.mName = iFilename;
				newRecord.mSize = iSize;
				mFonts.push_back(newRecord);
				record = &mFonts.back();
			}

			// Load up the face, which has the side effect of populating record->mYOffset and mHeight.
			FT_Face dummyFace;
			int error = FTC_Manager_LookupFace(mCacheManager, reinterpret_cast<FTC_FaceID>(record), &dummyFace);
			if (error)
				return false;

			oFaceId = reinterpret_cast<FTC_FaceID>(record);
			oOffset = record->mYOffset;
			oHeight = record->mHeight;
			return true;
		}

		bool LockBitmap(FTC_FaceID iFace, int iCharacter, FT_Int32 iFlags, FTC_SBit& oBit, FTC_Node& oNode)
		{
			FTC_ImageTypeRec type;
			type.flags = iFlags;
			FontRecord* fontRecord = reinterpret_cast<FontRecord*>(iFace);

#ifdef WIN32
			type.face_id = iFace;
			type.width = fontRecord->mSize;
			type.height = fontRecord->mSize;
			FT_UInt glyph = FTC_CMapCache_Lookup(mCharMapCache, iFace, 0, iCharacter);
#else
			type.font.face_id = iFace;
			type.font.pix_width = fontRecord->mSize;
			type.font.pix_height = fontRecord->mSize;
			FTC_CMapDescRec desc;
			desc.face_id = iFace;
			desc.type = FTC_CMAP_BY_INDEX;
			desc.u.index = FT_ENCODING_NONE;
			FT_UInt glyph = FTC_CMapCache_Lookup(mCharMapCache, &desc, iCharacter);
#endif
			int error = FTC_SBitCache_Lookup(mBitmapCache, &type, glyph, &oBit, &oNode);
			if (error)
			{
				return false;
			}

			return true;
		}

		void UnlockBitmap(FTC_Node iNode)
		{
			FTC_Node_Unref(iNode, mCacheManager);
		}

	private:
		FT_Library mLibrary;
		FTC_Manager mCacheManager;
		FTC_SBitCache mBitmapCache;
		FTC_CMapCache mCharMapCache;

		struct FontRecord
		{
			std::string mName;
			int mSize;
			int mYOffset;
			int mHeight;
		};

		std::list<FontRecord> mFonts;

		FontRecord* FindFont(const char* iFilename, int iSize)
		{
			for (std::list<FontRecord>::iterator i = mFonts.begin(); i != mFonts.end(); ++i)
			{
				if (i->mName == iFilename && i->mSize == iSize)
					return &*i;
			}
			return 0;
		}
	};


	IFontManager* CreateFontManager()
	{
		FontManager* manager = new FontManager();
		if (!manager->Initialise())
		{
			delete manager; 
			return 0;
		}	
		return manager;
	}

	void Font::Render(IDisplay* iDisplay, const char* iText, const Point& iStartPos)
	{
		int xPos = iStartPos.GetX();
		int yPos = iStartPos.GetY() + mYOffset;
		int num_chars = strlen(iText);

		iDisplay->BeginUpdate();
		int error;
		// Adjust the bitmaps so the first pixel is drawn on the left column.
		bool first = true;
		for (int i = 0; i < num_chars; ++i)
		{
			FTC_SBit bit;
			FTC_Node node;
			bool spaceHack = iText[i] == ' ';
			
			if (!mManager->LockBitmap(mFaceID, iText[i], spaceHack ? FT_LOAD_RENDER : FT_LOAD_RENDER | FT_LOAD_TARGET_MONO, bit, node))
				continue;

			if (!spaceHack)
			{
				if (first)
				{
					xPos -= bit->left;
					first = false;
				}
				DrawBitmap(iDisplay, bit, xPos + bit->left, yPos - bit->top); 
			}
			xPos += bit->xadvance;

			mManager->UnlockBitmap(node);
		}
		iDisplay->EndUpdate();
	}

	void Font::GetBoundingBox(const char* iText, const Point& iStartPos, Rectangle& oBounds) const
	{
		int xPos = iStartPos.GetX();
		int yPos = iStartPos.GetY() + mYOffset;
		int num_chars = strlen(iText);
		int error;
		oBounds.SetEmpty();

		// Adjust the bitmaps so the first pixel is drawn on the left column.
		bool first = true;
		for (int i = 0; i < num_chars; ++i)
		{
			FTC_SBit bit;
			FTC_Node node;
			bool spaceHack = iText[i] == ' ';

			if (!mManager->LockBitmap(mFaceID, iText[i], spaceHack ? FT_LOAD_RENDER : FT_LOAD_RENDER | FT_LOAD_TARGET_MONO, bit, node))
				continue;

			if (!spaceHack && first)
			{
				xPos -= bit->left;
				first = false;
			}

			Point topLeft(xPos + bit->left, yPos - bit->top);
			Point botRight(xPos + bit->left + bit->width, yPos - bit->top + bit->height);
			oBounds.ExtendToInclude(Rectangle(topLeft, botRight));

			xPos += bit->xadvance;
			mManager->UnlockBitmap(node);
		}
	}

	int Font::GetLineGapInPixels() const
	{
		return mLineHeight;
	}

}
