/*
    This file is part of the LCDGraphics library for the Weebox.

    LCDGraphics is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Foobar is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with LCDGraphics.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "HD61830Controller.h"


#include "IController.h"
#include "IDevice.h"
#include "Colour.h"


#include <iostream>


namespace LCDGraphics
{

	class HD61830Controller : public IController
	{
	public:
		HD61830Controller(IDevice* itDevice, int iWidth, int iHeight)
		{
			mDevice = itDevice;
			mWidth = iWidth;
			mHeight = iHeight;

			Poke(0x00, 0x32, "init"); // Initialise: Display on, master, graphics mode
			Poke(0x01, 0x77, "charsize");
			Poke(0x02, (mWidth / 8) - 1, "chars");
			Poke(0x03, mHeight - 1, "display duty");
			Poke(0x04, 0, "bottom line cursor");
			Poke(0x08, 0x00, "low order screen addr");
			Poke(0x09, 0x00, "high order screen addr");

			mConversionBuffer = new unsigned char[mWidth * mHeight / 8];
			mCurrentScreen = new unsigned char[mWidth * mHeight / 8];
			memset(mCurrentScreen, 0, mWidth * mHeight / 8);

			Poke(0x0a, 0x00, "low order cursor addr");
			Poke(0x0b, 0x00, "high order cursor addr");

			if (!mDevice->SetInstruction(0x0c))
				error("screen data", "instruction");

			if (!mDevice->SetData(mCurrentScreen, mHeight * mWidth / 8))
				error("screen data", "data");
		}
		~HD61830Controller()
		{
			delete[] mCurrentScreen;
			delete[] mConversionBuffer;
			delete mDevice;
		}

		virtual int GetHeight() const
		{
			return mHeight;
		}

		virtual int GetWidth() const
		{
			return mWidth;
		}

		virtual int GetNumColourChannels() const
		{
			return 1;
		}

		virtual int GetColourDepth() const
		{
			return 1;
		}

		virtual void SetScreen(Colour* iColourArray)
		{
			// TODO: conversion of the colour array in the display, not the drivers.
			ConvertBuffer(iColourArray);

			// Look for runs of bytes that differ from the current screen, and only send them.
			// If we find only a few 'same' bytes in a row, just send them as it's cheaper
			// than changing the cursor address.
			int hardwareCursorAddress = -1;
			int currentScreenAddress = 0;
			const int screenSize = mHeight * mWidth / 8;
			const int minSameRunLength = 4;
			while (currentScreenAddress < screenSize)
			{
				bool isSame = mConversionBuffer[currentScreenAddress] == mCurrentScreen[currentScreenAddress];
				int runLength;
				if (isSame)
				{
					for (runLength = 1; runLength < (screenSize - currentScreenAddress); ++runLength)
						if (mConversionBuffer[currentScreenAddress + runLength] != mCurrentScreen[currentScreenAddress + runLength])
							break;
				}
				else
				{
					for (runLength = 1; runLength < (screenSize - currentScreenAddress); ++runLength)
						if (mConversionBuffer[currentScreenAddress + runLength] == mCurrentScreen[currentScreenAddress + runLength])
							break;
				}
				if (!isSame || runLength < minSameRunLength)
				{
					if (hardwareCursorAddress != currentScreenAddress)
					{
						Poke(0x0a, currentScreenAddress & 0xff, "low order cursor addr");
						Poke(0x0b, (currentScreenAddress >> 8) & 0xff, "high order cursor addr");
						if (!mDevice->SetInstruction(0x0c))
							error("screen data", "instruction");					
						hardwareCursorAddress = currentScreenAddress;
					}
					if (!mDevice->SetData(mConversionBuffer + currentScreenAddress, runLength))
						error("screen data", "data");
					hardwareCursorAddress += runLength;
				}
				currentScreenAddress += runLength;
			}

			memcpy(mCurrentScreen, mConversionBuffer, screenSize);
		}

	private:
		void error(const char* iMessage, const char* iMessage2)
		{
			std::cerr << "Error in " << iMessage << " : " << iMessage2 << std::endl;
			exit(1);
		}

		void Poke(int iInstruction, int iData, const char *iMessage)
		{
			if (!mDevice->SetInstruction(iInstruction))
				error(iMessage, "instruction");
			if (!mDevice->SetData(iData))
				error(iMessage, "data");
		}

		void ConvertBuffer(Colour* iColourArray)
		{
			unsigned char* outputPtr = mConversionBuffer;
			Colour* colourPtr = iColourArray;
			for (int y = 0; y < mHeight; ++y)
			{
				for (int x = 0; x < mWidth; x+=8)
				{
					unsigned char outputByte = 0;
					for (int bit = 0; bit < 8; ++bit)
					{
						unsigned int red = colourPtr->GetRed();
						unsigned int blue = colourPtr->GetBlue();
						unsigned int green = colourPtr->GetGreen();
						// TODO: more intelligent than this!
						if ((red + blue + green) == 0)
							outputByte |= (1<<bit);
						colourPtr++;
					}
					*outputPtr++ = outputByte;
				}
			}
		}

	private:
		IDevice* mDevice;
		int mWidth;
		int mHeight;
		unsigned char* mConversionBuffer;
		unsigned char* mCurrentScreen;
	};

	IController* CreateHD61830Controller(IDevice* itDevice)
	{
		return new HD61830Controller(itDevice, 320, 64);
	}
}
