/*
    This file is part of the LCDGraphics library for the Weebox.

    LCDGraphics is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Foobar is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with LCDGraphics.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "Rectangle.h"

#include "Point.h"
#include "PointMaths.h"

#include <numeric>
#include <cassert>


using namespace LCDGraphics;


// ------ Constructors.

Rectangle::Rectangle()
{
	mTopLeft = Point(0,0);
	mBottomRight = Point(0,0);
}

Rectangle::Rectangle(const Point& iPoint1, const Point& iPoint2)
{
	int leftmostX = std::min(iPoint1.GetX(), iPoint2.GetX());
	int rightmostX = std::max(iPoint1.GetX(), iPoint2.GetX());
	int topmostY = std::min(iPoint1.GetY(), iPoint2.GetY());
	int bottommostY = std::max(iPoint1.GetY(), iPoint2.GetY());

	mTopLeft = Point(leftmostX, topmostY);
	mBottomRight = Point(rightmostX, bottommostY);
}

Rectangle::Rectangle(const Point& iTopLeft, int iWidth, int iHeight)
{
	assert(iWidth >= 0);
	if (iWidth < 0) iWidth = 0;
	assert(iHeight >= 0);
	if (iHeight < 0) iHeight = 0;
	mTopLeft = iTopLeft;
	mBottomRight = mTopLeft + Point(iWidth, iHeight);
}


// ------ Accessors.

int Rectangle::GetMinX() const
{
	return mTopLeft.GetX();
}

int Rectangle::GetMinY() const
{
	return mTopLeft.GetY();
}

int Rectangle::GetMaxX() const
{
	return mBottomRight.GetX();
}

int Rectangle::GetMaxY() const
{
	return mBottomRight.GetY();
}

int Rectangle::GetWidth() const
{
	return mBottomRight.GetX() - mTopLeft.GetX();
}

int Rectangle::GetHeight() const
{
	return mBottomRight.GetY() - mTopLeft.GetY();
}

void Rectangle::SetEmpty()
{
	mTopLeft = mBottomRight = Point(0, 0);
}

bool Rectangle::IsEmpty() const
{
	return mBottomRight.GetX() == mTopLeft.GetX()
		&& mBottomRight.GetY() == mTopLeft.GetY();
}

Point Rectangle::GetPoint(int iXpos, int iYpos) const
{
	int xPos, yPos;
	if (iXpos < 0)
		xPos = mTopLeft.GetX();
	else if (iXpos > 0)
		xPos = mBottomRight.GetX();
	else
		xPos = (mTopLeft.GetX() + mBottomRight.GetX()) / 2;

	if (iYpos < 0)
		yPos = mTopLeft.GetY();
	else if (iYpos > 0)
		yPos = mBottomRight.GetY();
	else
		yPos = (mTopLeft.GetY() + mBottomRight.GetY()) / 2;

	return Point(xPos, yPos);
}

void Rectangle::ClipTo(const Rectangle& iOther)
{
	int minX = GetMinX(), maxX = GetMaxX();
	int minY = GetMinY(), maxY = GetMaxY();

	if (minX < iOther.GetMinX())
		minX = iOther.GetMinX();
	if (minX > iOther.GetMaxX())
		minX = iOther.GetMaxX();
	if (maxX < iOther.GetMinX())
		maxX = iOther.GetMinX();
	if (maxX > iOther.GetMaxX())
		maxX = iOther.GetMaxX();

	if (minY < iOther.GetMinY())
		minY = iOther.GetMinY();
	if (minY > iOther.GetMaxY())
		minY = iOther.GetMaxY();
	if (maxY < iOther.GetMinY())
		maxY = iOther.GetMinY();
	if (maxY > iOther.GetMaxY())
		maxY = iOther.GetMaxY();

	mTopLeft = Point(minX, minY);
	mBottomRight = Point(maxX, maxY);
}

void Rectangle::ExtendToInclude(const Rectangle& iOther)
{
	if (IsEmpty())
		*this = iOther;
	else
	{
		int minX = GetMinX(), maxX = GetMaxX();
		int minY = GetMinY(), maxY = GetMaxY();

		if (iOther.GetMinX() < minX)
			minX = iOther.GetMinX();
		if (iOther.GetMaxX() > maxX)
			maxX = iOther.GetMaxX();

		if (iOther.GetMinY() < minY)
			minY = iOther.GetMinY();
		if (iOther.GetMaxY() > maxY)
			maxY = iOther.GetMaxY();

		mTopLeft = Point(minX, minY);
		mBottomRight = Point(maxX, maxY);
	}
}

bool Rectangle::Contains(const Point& iPoint) const
{
	return (iPoint.GetX() >= mTopLeft.GetX()
		&& iPoint.GetX() < mBottomRight.GetX()
		&& iPoint.GetY() >= mTopLeft.GetY()
		&& iPoint.GetY() < mBottomRight.GetY());
}

bool Rectangle::Contains(const Rectangle& iOther) const
{
	// TODO: this is actually wrong, need to do proper outcoding etc
	return Contains(iOther.GetPoint(-1, -1))
		|| Contains(iOther.GetPoint(-1,  1))
		|| Contains(iOther.GetPoint( 1, -1))
		|| Contains(iOther.GetPoint( 1,  1));
}

void Rectangle::Grow(int iPixels)
{
	mTopLeft = Point(mTopLeft.GetX() - iPixels, mTopLeft.GetY() - iPixels);
	mBottomRight = Point(mBottomRight.GetX() + iPixels, mBottomRight.GetY() + iPixels);
}

void Rectangle::Shrink(int iPixels)
{
	if (iPixels >= GetHeight() || iPixels >= GetWidth())
		SetEmpty();
	else
		Grow(-iPixels);
}

void Rectangle::SetTopLeft(const Point& iTopLeft)
{
	int prevWidth = GetWidth();
	int prevHeight = GetHeight();
	mTopLeft = iTopLeft;
	mBottomRight = mTopLeft + Point(prevWidth, prevHeight);
}
