/*
    This file is part of the LCDGraphics library for the Weebox.

    LCDGraphics is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Foobar is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with LCDGraphics.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "WindowsController.h"

#include "IController.h"
#include "Colour.h"

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <deque>
#include <assert.h>


namespace LCDGraphics
{

	class WindowsController : public IController, public IKeySource
	{
	public:
		WindowsController(int iWidth, int iHeight)
		{
			mHeight = iHeight;
			mWidth = iWidth;
			mLCDWindow = 0;

			InitializeCriticalSection(&mBitmapCritSection);
			InitializeCriticalSection(&mKeyCritSection);

			// Create the backbuffer.
			mBitmap = CreateBitmap(iWidth, iHeight, 1, 1, NULL);

			DWORD threadId;
			CreateThread(NULL, 64 * 1024, &WindowsController::ThreadRoutine, this, 0, &threadId);

			// Appalling way to wait for the thread to initialise
			while (mLCDWindow == 0)
				Sleep(1);
		}

		~WindowsController()
		{
			DeleteObject(&mBitmap);
			DeleteCriticalSection(&mBitmapCritSection);
			DeleteCriticalSection(&mKeyCritSection);
		}

		int GetHeight() const
		{
			return mHeight;
		}
		int GetWidth() const
		{
			return mWidth;
		}
		int GetNumColourChannels() const
		{
			return 1;
		}
		int GetColourDepth() const
		{
			return 1;
		}

		virtual void SetScreen(Colour* iColourArray)
		{
			int nBytes = mWidth * mHeight / 8;
			char* arrayBytes = new char[nBytes];
			char* arrayPtr = arrayBytes;
			Colour* colourPtr = iColourArray;
			for (int y = 0; y < mHeight; ++y)
			{
				for (int x = 0; x < mWidth; x += 8)
				{
					unsigned char byte = 0;
					unsigned char value = 0x80;
					for (int bit = 0; bit < 8; ++bit)
					{
						unsigned int red = colourPtr->GetRed();
						unsigned int blue = colourPtr->GetBlue();
						unsigned int green = colourPtr->GetGreen();
						// TODO: more intelligent than this!
						if ((red + blue + green) != 0)
							byte |= value;
						value >>= 1;
						++colourPtr;
					}
					*arrayPtr++ = byte;
				}
			}
			EnterCriticalSection(&mBitmapCritSection);
			LONG numSet = SetBitmapBits(mBitmap, nBytes, arrayBytes);
			LeaveCriticalSection(&mBitmapCritSection);
			if (mLCDWindow)
				RedrawWindow(mLCDWindow, NULL, NULL, RDW_INVALIDATE);
			delete[] arrayBytes;
		}


		static LRESULT CALLBACK WindowProc(HWND iHwnd, UINT iMsg, WPARAM iWparam, LPARAM iLparam)
		{
			static WindowsController* theWindowsController = 0;
			switch (iMsg)
			{
			case WM_CREATE:
				{
					CREATESTRUCT* cs = reinterpret_cast<CREATESTRUCT*>(iLparam);
					theWindowsController = reinterpret_cast<WindowsController*>(cs->lpCreateParams);
				}
				break;
			case WM_PAINT:
				{
					RECT rect;
					if (GetUpdateRect(iHwnd, &rect, FALSE))
					{
						PAINTSTRUCT paintStruct;
						HDC dc = BeginPaint(iHwnd, &paintStruct);
						HDC dcSrc = CreateCompatibleDC(dc);
						EnterCriticalSection(&theWindowsController->mBitmapCritSection);
						SelectObject(dcSrc, theWindowsController->mBitmap);
						BitBlt(dc, 0, 0, 
							theWindowsController->mWidth, theWindowsController->mHeight, 
							dcSrc, 0, 0, SRCCOPY);
						LeaveCriticalSection(&theWindowsController->mBitmapCritSection);
						DeleteDC(dcSrc);
						EndPaint(iHwnd, &paintStruct);
					}
				}
				break;
			case WM_KEYDOWN:
				{
					EnterCriticalSection(&theWindowsController->mKeyCritSection);
					theWindowsController->mKeys.push_back((char)iWparam);
					LeaveCriticalSection(&theWindowsController->mKeyCritSection);
				}
				break;
			default:
				return DefWindowProc(iHwnd, iMsg, iWparam, iLparam);
			}

			return 0;
		}

		virtual int GetKeypress()
		{
			int keyCode = -1;
			EnterCriticalSection(&mKeyCritSection);
			if (!mKeys.empty())
			{
				keyCode = mKeys.front();
				mKeys.pop_front();
			}
			LeaveCriticalSection(&mKeyCritSection);
			return keyCode;
		}

		static DWORD CALLBACK ThreadRoutine(void* iParam)
		{
			WindowsController* _this = reinterpret_cast<WindowsController*>(iParam);
			// Create the window.
			WNDCLASSEX windowClass;
			windowClass.cbSize = sizeof(WNDCLASSEX);
			windowClass.style = 0;
			windowClass.lpfnWndProc = &WindowsController::WindowProc;
			windowClass.cbClsExtra = 0;
			windowClass.cbWndExtra = 0;
			windowClass.hInstance = GetModuleHandle(NULL);
			windowClass.hIcon = NULL;
			windowClass.hCursor = LoadCursor(NULL, IDC_ARROW);
			windowClass.hbrBackground = (HBRUSH)(COLOR_WINDOW+1);
			windowClass.lpszMenuName = NULL;
			windowClass.lpszClassName = "LCDGraphics::WindowsController";
			windowClass.hIconSm = NULL;
			ATOM atom = RegisterClassEx(&windowClass);

			DWORD style = WS_OVERLAPPEDWINDOW;

			RECT rect;
			rect.top = rect.left = 0;
			rect.bottom = _this->mHeight;
			rect.right = _this->mWidth;
			AdjustWindowRect(&rect, style, FALSE);

			_this->mLCDWindow = CreateWindowEx(0, 
				(LPCSTR)atom,
				"LCD Display", 
				style, 
				CW_USEDEFAULT, CW_USEDEFAULT, 
				rect.right - rect.left, rect.bottom - rect.top,
				NULL, NULL, 
				GetModuleHandle(NULL),
				_this);

			ShowWindow(_this->mLCDWindow, SW_SHOWNORMAL);

			MSG msg;
			while(GetMessage(&msg, NULL, 0, 0) > 0)
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
			return 0;
		}
	private:
		HWND mLCDWindow;
		int mWidth;
		int mHeight;
		HBITMAP mBitmap;
		CRITICAL_SECTION mBitmapCritSection;

		CRITICAL_SECTION mKeyCritSection;
		std::deque<char> mKeys;
	};

	static WindowsController* theController = 0;

	IController* CreateWindowsController()
	{
		assert(theController == 0);
		theController = new WindowsController(320, 64);
		return theController;
	}

	extern IKeySource* GetKeySource()
	{
		return theController;
	}
}
